using System;
using System.Runtime.InteropServices;
using System.Text;

namespace Sony
{
	namespace NP
	{
        public class Friends
		{
			// Friends
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxFriendsListIsBusy();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxRefreshFriendsList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxLockFriendsList();
			[DllImport("UnityNpToolkit")]
			private static extern void PrxUnlockFriendsList();
			[DllImport("UnityNpToolkit")]
			private static extern int PrxGetFriendCount();
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxGetFriend(int index, out Friend frnd);
			
			[DllImport("UnityNpToolkit")]
			private static extern bool PrxFriendsGetLastError(out ResultCode result);
			
			public static bool GetLastError(out ResultCode result)
			{
				PrxFriendsGetLastError(out result);
				return result.lastError == ErrorCode.NP_OK;
			}

			public enum EnumNpOnlineStatus
			{
				ONLINE_STATUS_UNKNOWN = 0,        // Unknown
				ONLINE_STATUS_OFFLINE = 1,        // Offline
				ONLINE_STATUS_AFK = 2,            // Not Available
				ONLINE_STATUS_ONLINE = 3,         // Online
			};

			public enum EnumNpPresenceType
			{
				IN_GAME_PRESENCE_TYPE_UNKNOWN = -1,    // Unknown
				IN_GAME_PRESENCE_TYPE_NONE = 0,        // None
				IN_GAME_PRESENCE_TYPE_DEFAULT = 1,     // Default
				IN_GAME_PRESENCE_TYPE_GAME_JOINING = 2,// Joining game (including invitation data)
				IN_GAME_PRESENCE_TYPE_GAME_JOINING_ONLY_FOR_PARTY = 3, // Joining game only for party
				IN_GAME_PRESENCE_TYPE_JOIN_GAME_ACK = 4, // Join Game Ack
				IN_GAME_PRESENCE_TYPE_MAX = 5          // Maximum
			};

			const int NP_ONLINEID_MAX_LENGTH = 16;              // Max length of the users Online ID string.
			const int IN_GAME_PRESENCE_STATUS_SIZE_MAX = 192;   // Max length of the online presence string.
			const int IN_GAME_PRESENCE_DATA_SIZE_MAX = 128;     // Max length of online presence data array.

			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct Friend
			{
				IntPtr _npID;								// NpID, contains data corresponding to the SceNpID structure, can be used as id to send to when messaging.
				int npIDSize;								// Size of the npID byte array.
				IntPtr _npOnlineID;                         // NpID online string, 16 bytes plus terminator.
				public EnumNpOnlineStatus npOnlineStatus;   // Online status.
				IntPtr _npPresenceTitle;                    // 128 bytes.
				public int npPresenceSdkVersion;
				public EnumNpPresenceType npPresenceType;   // InGame presence type, SceNpBasicInGamePresenceType
				IntPtr _npPresenceStatus;                   // Game play status, 192 characters.
				IntPtr _npComment;							// Comment - 64 characters	-	 PS3 Only
				IntPtr _npPresenceData;                     // Freely defined data, 128 bytes.
				public int npPresenceDataSize;

				public byte[] npID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
				public string npOnlineID { get { return Marshal.PtrToStringAnsi(_npOnlineID); } }
				public string npPresenceTitle { get { return Marshal.PtrToStringAnsi(_npPresenceTitle); } }
				public string npPresenceStatus { get { return Marshal.PtrToStringAnsi(_npPresenceStatus); } }
				public string npComment { get { return Marshal.PtrToStringAnsi(_npComment); } }
				public byte[] npPresenceData
				{
					get
					{
						byte[] bytes = new byte[IN_GAME_PRESENCE_DATA_SIZE_MAX];
						Marshal.Copy(_npPresenceData, bytes, 0, IN_GAME_PRESENCE_DATA_SIZE_MAX);
						return bytes;
					}
				}
			};

			public static event Messages.EventHandler OnFriendsListUpdated;
			public static event Messages.EventHandler OnFriendsPresenceUpdated;
			public static event Messages.EventHandler OnGotFriendsList;
			public static event Messages.EventHandler OnFriendsListError;

			public static ErrorCode RequestFriendsList()
			{
				return PrxRefreshFriendsList();
			}

			public static bool FriendsListIsBusy()
			{
				return PrxFriendsListIsBusy();
			}

			public static Friend[] GetCachedFriendsList()
			{
				PrxLockFriendsList();
				Friend[] friends = new Friend[PrxGetFriendCount()];
				for (int i = 0; i < PrxGetFriendCount(); i++)
				{
					PrxGetFriend(i, out friends[i]);
				}
				PrxUnlockFriendsList();

				return friends;
			}

			public static bool ProcessMessage(Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case Messages.MessageType.kNPToolKit_FriendsListUpdated:
						if (OnFriendsListUpdated != null) OnFriendsListUpdated(msg);
						return true;
					case Messages.MessageType.kNPToolkit_FriendsPresenceUpdated:
						if (OnFriendsPresenceUpdated != null) OnFriendsPresenceUpdated(msg);
						return true;
					case Messages.MessageType.kNPToolKit_GotFriendsList:
						if (OnGotFriendsList != null) OnGotFriendsList(msg);
						return true;

					case Messages.MessageType.kNPToolKit_FriendsListError:
						if (OnFriendsListError != null) OnFriendsListError(msg);
						return true;
				}

				return false;
			}
		}
	} // NP
} // Sony
